// Модуль записи-чтения блока данных в ИМС часов-календаря  PCF8563 
// Запись и тчение нельзя производить одновременно,   поскольку для них используется одна и та же линия данных
module  PCF8563_wr_rd(

     // сигналы обмена с остальной конфигурацией ПЛИС 
     input clk, 	
	input nrst,   
  	input PCF8563_write_start, // сигнал  старта  записи    
	input PCF8563_read_start, // сигнал  старта  чтения	
  
	input [7:0] PCF8563_write_data_1,  //------------------------- 
     input [7:0] PCF8563_write_data_2,  //   данные записи 
	input [7:0] PCF8563_write_data_3,  //  
     input [7:0] PCF8563_write_data_4,  //    регистров RTC  PCF8563
     input [7:0] PCF8563_write_data_5,  // 	
     input [7:0] PCF8563_write_data_6,  //    
     input [7:0] PCF8563_write_data_7,  //------------------------- 
 		
	output [7:0] PCF8563_read_data_1,  //------------------------- 
	output [7:0] PCF8563_read_data_2,  //
	output [7:0] PCF8563_read_data_3,  //  данные чтения  
	output [7:0] PCF8563_read_data_4,  //
	output [7:0] PCF8563_read_data_5,  //  регистров RTC  PCF8563
	output [7:0] PCF8563_read_data_6,  //
	output [7:0] PCF8563_read_data_7,  //-------------------------	
				
	output write_final_imp,  // импульс финала записи 
     output read_final_imp,  // импульс финала чтения
   
	  // сигналы обмена с ИМС PCF8563
     output  PCF8563_SCL,  // синхросигнал PCF8563
     inout PCF8563_SDA   // двунаправленная линия данных для чтения/записи PCF8563
	); 
	

//-----------------------------------------------------------------------------------------
// Сигналы и регистры записи 
//-----------------------------------------------------------------------------------------
wire write_start_imp;  // импульс старта записи 

reg  write_I2C = 0; // признак длящейся записи I2C 

reg  write_final = 0; // сигнал завершения записи 

// выходы двухступенчатого триггера формирователя положительного импульса  длительностью 1 период clk 
reg Q1 = 0; //
reg Q2 = 0; //
reg R1 = 0; //
reg R2 = 0; //

    
//-----------------------------------------------------------------------------------------  
// Сигналы и регистры чтения 	  
//-----------------------------------------------------------------------------------------
wire read_start_imp;  // импульс старта чтения 
	  
reg [7:0] reg_PCF8563_read_data_1 = 0; // регистр данных чтения: секунды
reg [7:0] reg_PCF8563_read_data_2 = 0; // регистр данных чтения: минуты 
reg [7:0] reg_PCF8563_read_data_3 = 0; // регистр данных чтения: часы
reg [7:0] reg_PCF8563_read_data_4 = 0; // регистр данных чтения: числа месяца 
reg [7:0] reg_PCF8563_read_data_5 = 0; // регистр данных чтения: дни недели 
reg [7:0] reg_PCF8563_read_data_6 = 0; // регистр данных чтения: месяцы 
reg [7:0] reg_PCF8563_read_data_7 = 0; // регистр данных чтения:	годы   
	 
reg  read_I2C = 0; // признак длящегося чтения I2C
	
reg  read_final = 0; // сигнал завершения чтения 

// выходы двухступенчатого триггера формирователя положительного импульса  длительностью 1 период clk 
reg P1 = 0; //
reg P2 = 0; //
reg M1 = 0; //
reg M2 = 0; //
  
  
//-----------------------------------------------------------------------------------------
// Общие сигналы, регистры  и  параметры
//-----------------------------------------------------------------------------------------

reg reg_PCF8563_SCL = 1'bz; // регистр управления линией SCL 

reg reg_PCF8563_SDA = 1'bz; // регистр управления линией SDA

reg [31:0] counter_T_bit = 0; // счетчик отсчета интервала между  передаваемыми битами 

parameter   GLOBAL_CLOCK  = 50_000_000;  // частота глобального синхросигнала, Гц 

parameter PCF8563_RATE = 500_000;  //  1/3  частоты синхросигнала PCF8563
parameter  T = GLOBAL_CLOCK / PCF8563_RATE;  // 1/2 интервала между  битами в байте 


parameter  ADDR_SLAVE_PCF8563 = 7'b1010_001;  // 7-разрядный адрес ведомого   PCF8563   на шине I2C
parameter  WORD_ADDR_WRITE = 8'b0000_0010; // 8-разрядный начальный адрес регистра для  записи   по шине I2C 
parameter  WORD_ADDR_READ  = 8'b0000_0010; // 8-разрядный начальный адрес регистра для  чтения   по шине I2C

parameter  WRITE_BIT = 0;  //  бит (признак) записи 
parameter  READ_BIT = 1'bz;  // бит (признак) чтения (отпускаем линию, чтобы она установилась от резистора подтяжки к плюсу)

//-------------------------------------------------------------------------------------------------
//-------------------------------------------------------------------------------------------------
//-------------------------------------------------------------------------------------------------

//-----------------------------------------------------------------------------------------
// Код, относящийся к записи 
//-----------------------------------------------------------------------------------------
// формирователь импульса  старта записи  
// длительностью один период clk по фронту  сигнала старта записи  	  
always@(posedge (~clk)) 	  
  begin  
     Q1 <= PCF8563_write_start;
	Q2 <= Q1; 
  end  
 
assign write_start_imp = Q1 & (~Q2);

// формирователь импульса  финала записи  
// длительностью один период clk по фронту  сигнала финала записи  	  
always@(posedge (~clk)) 	  
  begin  
     R1 <= write_final;
     R2 <= R1; 
  end  
 
assign write_final_imp = R1 & (~R2);
						
// Генерация состояния длящейся записи по обнаружению стартового импульса 
always @(posedge clk)      
begin 
 if(!nrst) //  реакция системы на сброс 
    begin 
      write_I2C <= 0;   //  сбросить признак длящейся записи
	 end 
 else if (write_start_imp)  // если  обнаружен   стартовый импульс    
	 begin 
	   write_I2C <= 1;	// установить признак длящейся записи 
      end 
 else if (write_final_imp)	 //  если  запись завершена
	 begin
        write_I2C <= 0;   //  сбросить признак длящейся записи
    end		
end
	 	 

//-----------------------------------------------------------------------------------------
// Код, относящийся к чтению
//-----------------------------------------------------------------------------------------
// формирователь импульса  старта чтения  
// длительностью один период clk по фронту  сигнала старта чтения  	  
always@(posedge (~clk)) 	  
  begin  
     P1 <= PCF8563_read_start;
	P2 <= P1; 
  end  
 
assign read_start_imp = P1 & (~P2);

// формирователь импульса  финала чтения  
// длительностью один период clk по фронту  сигнала финала чтения  	  
always@(posedge (~clk)) 	  
  begin  
     M1 <= read_final;
	M2 <= M1; 
  end  
 
assign read_final_imp = M1 & (~M2);
						
// Генерация состояния длящейся чтения по обнаружению стартового импульса 
always @(posedge clk)      
begin 
 if(!nrst) //  реакция системы на сброс 
    begin 
      read_I2C <= 0;   //  сбросить признак длящейся чтения
	 end 
 else if (read_start_imp)  // если  обнаружен   стартовый импульс    
	 begin 
	    read_I2C <= 1;	// установить признак длящейся чтения 
      end 
 else if (read_final_imp)	 //  если  чтение завершено
	 begin
         read_I2C <= 0;   //  сбросить признак длящейся чтения
    end		
end
		 
// Управление выходами данных  чтения от регистров данных чтения  
assign  PCF8563_read_data_1 = reg_PCF8563_read_data_1;
assign  PCF8563_read_data_2 = reg_PCF8563_read_data_2;
assign  PCF8563_read_data_3 = reg_PCF8563_read_data_3;
assign  PCF8563_read_data_4 = reg_PCF8563_read_data_4;
assign  PCF8563_read_data_5 = reg_PCF8563_read_data_5;
assign  PCF8563_read_data_6 = reg_PCF8563_read_data_6;
assign  PCF8563_read_data_7 = reg_PCF8563_read_data_7;
		
		
		
//-----------------------------------------------------------------------------------------
// Общий код 
//-----------------------------------------------------------------------------------------

// Управление синхросигналом 
assign  PCF8563_SCL = reg_PCF8563_SCL; 

// Управление  линией данных
assign  PCF8563_SDA = reg_PCF8563_SDA; 

//-----------------------------------------------------------------------------------------
// Запись или чтение блока данных в PCF8563	
always @(posedge clk)
begin 
 if(!nrst) //  реакция системы на сброс 
    begin 
	   counter_T_bit <= 0; // обнулить счетчик тактирования
	   write_final <= 0; // сбросить сигнал финала записи
	   read_final <= 0; // сбросить сигнал финала чтения 	 
    end

 // Процедура записи	 
 else  if (write_I2C)  //  если установлен признак длящейся записи  
    begin
       counter_T_bit <= counter_T_bit + 1;
		 case (counter_T_bit)  
		              //    выполнение  условия START
		              1: begin  reg_PCF8563_SCL <= 1'bz;  reg_PCF8563_SDA <= 1'bz; end  //  отпустить SDA, SCL   
				    T: begin  reg_PCF8563_SDA <= 0; end  //  сбросить SDA
				(2*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			         // условие START  выполнено, SDA, SCL в низком уровне   	
                        // передача адреса ведомого устройства  с признаком записи  
				(3*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[6]) ? (1'bz) : 0;  end  // передача на шину бита   
				(4*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
				(5*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL                                  				    
				(6*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[5]) ? (1'bz) : 0;  end  // передача на шину бита   
				(7*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
				(8*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				   
				(9*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (10*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (11*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				
			    (12*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (13*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (14*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  
			    (15*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (16*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (17*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 			    
			    (18*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (19*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (20*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (21*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (22*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (23*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (24*T): begin  reg_PCF8563_SDA <= WRITE_BIT;  end  // передача на шину бита записи   
			    (25*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (26*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (27*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (28*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (29*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
                        // передача адреса первого регистра для записи  
			    (30*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (31*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (32*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			    (33*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (34*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (35*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			    (36*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (37*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (38*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (39*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (40*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (41*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (42*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (43*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (44*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (45*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (46*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (47*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			    (48*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (49*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (50*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (51*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_WRITE[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (52*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (53*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (54*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (55*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (56*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			         // передача даных для записи первого адресованного регистра  
			    (57*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (58*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (59*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			    (60*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (61*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (62*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			    (63*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (64*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (65*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (66*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (67*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (68*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (69*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (70*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (71*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (72*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (73*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (74*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			    (75*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (76*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (77*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (78*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_1[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (79*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (80*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (81*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (82*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (83*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
			         // передача даных для записи второго адресованного регистра  
			    (84*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (85*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (86*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			    (87*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (88*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (89*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			    (90*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (91*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (92*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (93*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (94*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (95*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (96*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (97*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (98*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (99*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (100*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (101*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (102*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (103*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (104*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (105*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_2[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (106*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (107*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (108*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (109*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (110*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL					
			         // передача даных для записи третьего адресованного регистра  
			   (111*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (112*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (113*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			   (114*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (115*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (116*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			   (117*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (118*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (119*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (120*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (121*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (122*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (123*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (124*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (125*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			   (126*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (127*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (128*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (129*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (130*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (131*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (132*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_3[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (133*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (134*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (135*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (136*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (137*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL						   
			         // передача даных для записи четвертого адресованного регистра  
			   (138*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (139*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (140*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			   (141*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (142*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (143*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			   (144*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (145*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (146*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (147*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (148*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (149*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (150*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (151*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (152*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			   (153*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (154*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (155*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (156*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (157*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (158*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (159*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_4[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (160*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (161*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (162*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (163*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (164*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL					
			         // передача даных для записи пятого адресованного регистра  
			   (165*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (166*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (167*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			   (168*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (169*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (170*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			   (171*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (172*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (173*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (174*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (175*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (176*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (177*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (178*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (179*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			   (180*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (181*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (182*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (183*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (184*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (185*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (186*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_5[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (187*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (188*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (189*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (190*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (191*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
			         // передача даных для записи шестого адресованного регистра  
			   (192*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (193*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (194*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			   (195*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (196*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (197*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			   (198*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (199*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (200*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (201*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (202*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (203*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (204*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (205*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (206*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			   (207*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (208*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (209*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (210*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (211*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (212*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (213*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_6[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (214*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (215*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (216*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (217*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (218*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
			         // передача даных для записи седьмого адресованного регистра  
			   (219*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (220*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (221*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			   (222*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (223*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (224*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			   (225*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (226*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (227*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (228*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (229*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (230*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (231*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (232*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (233*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			   (234*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (235*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (236*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			   (237*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (238*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (239*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (240*T): begin  reg_PCF8563_SDA <= (PCF8563_write_data_7[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			   (241*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (242*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			   (243*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			   (244*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (245*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL		
		              //    выполнение  условия STOP 			
			   (246*T): begin  reg_PCF8563_SDA <= 0; end  //  сбросить SDA 			  
			   (247*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   			
			   (248*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA			
                        // условие STOP  выполнено 			
			   (249*T): begin                  
			               write_final <= 1; // установить сигнал финала записи
					  end 
		 endcase
    end
	 
 // Процедура чтения 
 else  if (read_I2C)  //  если установлен признак длящегося чтения  
    begin
       counter_T_bit <= counter_T_bit + 1;
		 case (counter_T_bit)  
		              //    выполнение  условия START
		              1: begin  reg_PCF8563_SCL <= 1'bz;  reg_PCF8563_SDA <= 1'bz; end  //  отпустить SDA, SCL   
				    T: begin  reg_PCF8563_SDA <= 0; end  //  сбросить SDA
				(2*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			         // условие START  выполнено, SDA, SCL в низком уровне   	
                        // передача адреса ведомого устройства с признаком записи  
				(3*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[6]) ? (1'bz) : 0;  end  // передача на шину бита   
				(4*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
				(5*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL                                  				    
				(6*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[5]) ? (1'bz) : 0;  end  // передача на шину бита   
				(7*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
				(8*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				   
				(9*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (10*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (11*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				
			    (12*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (13*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (14*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  
			    (15*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (16*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (17*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 			    
			    (18*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (19*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (20*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (21*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (22*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (23*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (24*T): begin  reg_PCF8563_SDA <= WRITE_BIT;  end  // передача на шину бита записи   
			    (25*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (26*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (27*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (28*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (29*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
                        // передача адреса первого регистра для  чтения   
			    (30*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[7]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (31*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (32*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL       			
			    (33*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (34*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (35*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			 
			    (36*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (37*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (38*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (39*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (40*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (41*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (42*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (43*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (44*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (45*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (46*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (47*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			    
			    (48*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (49*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (50*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (51*T): begin  reg_PCF8563_SDA <= (WORD_ADDR_READ[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (52*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (53*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
			    (54*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (55*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (56*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
		              //    выполнение  условия повторный START
		         (57*T): begin  reg_PCF8563_SCL <= 1'bz; reg_PCF8563_SDA <= 1'bz; end  //  отпустить SCL, SDA  
			    (58*T): begin  reg_PCF8563_SDA <= 0; end  //  сбросить SDA 
			    (59*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  			    
			         // условие повторный START  выполнено, SDA, SCL в низком уровне  
                        // передача адреса ведомого устройства  с признаком чтения  
			    (60*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[6]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (61*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (62*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL                                  				    
			    (63*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[5]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (64*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (65*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				   
			    (66*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[4]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (67*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (68*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  				
			    (69*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[3]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (70*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (71*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL  
			    (72*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[2]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (73*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (74*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 			    
			    (75*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[1]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (76*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (77*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (78*T): begin  reg_PCF8563_SDA <= (ADDR_SLAVE_PCF8563[0]) ? (1'bz) : 0;  end  // передача на шину бита   
			    (79*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (80*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL 
			    (81*T): begin  reg_PCF8563_SDA <= READ_BIT;  end  // передача на шину бита чтения   
			    (82*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (83*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
			    (84*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   анализа бита подтверждения      
			    (85*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			    (86*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL
	                 // чтение первого адресованного регистра 
			    (87*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			    (88*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			    (89*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			    (90*T): begin   end  // пауза   
			    (91*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			    (92*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			    (93*T): begin   end  // пауза   
			    (94*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			    (95*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			    (96*T): begin   end  // пауза   
			    (97*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			    (98*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			    (99*T): begin   end  // пауза   
			   (100*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (101*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (102*T): begin   end  // пауза   
			   (103*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (104*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (105*T): begin   end  // пауза   
			   (106*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (107*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (108*T): begin   end  // пауза   
			   (109*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (110*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_1[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (111*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (112*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (113*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL			
	                // чтение второго адресованного регистра 
			   (114*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (115*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (116*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (117*T): begin   end  // пауза   
			   (118*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (119*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (120*T): begin   end  // пауза   
			   (121*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (122*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (123*T): begin   end  // пауза   
			   (124*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (125*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (126*T): begin   end  // пауза   
			   (127*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (128*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (129*T): begin   end  // пауза   
			   (130*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (131*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (132*T): begin   end  // пауза   
			   (133*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (134*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (135*T): begin   end  // пауза   
			   (136*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (137*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_2[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (138*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (139*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (140*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
	                // чтение третьего адресованного регистра 
			   (141*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (142*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (143*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (144*T): begin   end  // пауза   
			   (145*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (146*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (147*T): begin   end  // пауза   
			   (148*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (149*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (150*T): begin   end  // пауза   
			   (151*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (152*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (153*T): begin   end  // пауза   
			   (154*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (155*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (156*T): begin   end  // пауза   
			   (157*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (158*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (159*T): begin   end  // пауза   
			   (160*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (161*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (162*T): begin   end  // пауза   
			   (163*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (164*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_3[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (165*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (166*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (167*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
	                // чтение четвертого адресованного регистра 
			   (168*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (169*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (170*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (171*T): begin   end  // пауза   
			   (172*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (173*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (174*T): begin   end  // пауза   
			   (175*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (176*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (177*T): begin   end  // пауза   
			   (178*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (179*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (180*T): begin   end  // пауза   
			   (181*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (182*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (183*T): begin   end  // пауза   
			   (184*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (185*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (186*T): begin   end  // пауза   
			   (187*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (188*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (189*T): begin   end  // пауза   
			   (190*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (191*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_4[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (192*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (193*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (194*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL	
	                // чтение пятого адресованного регистра 
			   (195*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (196*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (197*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (198*T): begin   end  // пауза   
			   (199*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (200*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (201*T): begin   end  // пауза   
			   (202*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (203*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (204*T): begin   end  // пауза   
			   (205*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (206*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (207*T): begin   end  // пауза   
			   (208*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (209*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (210*T): begin   end  // пауза   
			   (211*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (212*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (213*T): begin   end  // пауза   
			   (214*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (215*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (216*T): begin   end  // пауза   
			   (217*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (218*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_5[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (219*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (220*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (221*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL	
	                // чтение шестого адресованного регистра 
			   (222*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (223*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (224*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (225*T): begin   end  // пауза   
			   (226*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (227*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (228*T): begin   end  // пауза   
			   (229*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (230*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (231*T): begin   end  // пауза   
			   (232*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (233*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (234*T): begin   end  // пауза   
			   (235*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (236*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (237*T): begin   end  // пауза   
			   (238*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (239*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (240*T): begin   end  // пауза   
			   (241*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (242*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (243*T): begin   end  // пауза   
			   (244*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (245*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_6[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (246*T): begin  reg_PCF8563_SDA <= 0;  end  // сбросить SDA  для   выдачи подтверждения      
			   (247*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (248*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				

	                // чтение седьмого адресованного регистра 
			   (249*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для  чтения     
			   (250*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (251*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[7] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (252*T): begin   end  // пауза   
			   (253*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (254*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[6] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA 	  
			   (255*T): begin   end  // пауза   
			   (256*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (257*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[5] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (258*T): begin   end  // пауза   
			   (259*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (260*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[4] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (261*T): begin   end  // пауза   
			   (262*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (263*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[3] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA
			   (264*T): begin   end  // пауза   
			   (265*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (266*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[2] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (267*T): begin   end  // пауза   
			   (268*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (269*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[1] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (270*T): begin   end  // пауза   
			   (271*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL  	
			   (272*T): begin  reg_PCF8563_SCL <= 0;  reg_PCF8563_read_data_7[0] <= PCF8563_SDA; end  // сбросить SCL, прочитать бит с шины SDA			    
			   (273*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA  для   выдачи неподтверждения      
			   (274*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   
			   (275*T): begin  reg_PCF8563_SCL <= 0; end  //  сбросить SCL				
		              //    выполнение  условия STOP 			
			   (276*T): begin  reg_PCF8563_SDA <= 0; end  //  сбросить SDA 			  
			   (277*T): begin  reg_PCF8563_SCL <= 1'bz;  end  //  отпустить SCL   			
			   (278*T): begin  reg_PCF8563_SDA <= 1'bz;  end  // отпустить SDA			
                        // условие STOP  выполнено 			
			   (279*T): begin                  
			               read_final <= 1; // установить сигнал финала чтения
					  end 	  
		 endcase	  
    end
 else   // если ни чтение, ни запись не ведутся 
    begin
	   counter_T_bit <= 0; // обнулить счетчик тактирования
	   write_final <= 0; // сбросить сигнал финала записи
	   read_final <= 0; // сбросить сигнал финала чтения
    end
end 
	

endmodule	